/* 
 Name:
	Create3DPlatonicSolid
 Version:
	1.0 (15 August 2010)
 Author:
	Charles Bordenave
 Description:  
	This script creates 3D regular convex polyhedra known as Platonic Solids.
 Usage:
	In After Effects CS4 or later, run the script
	Specify the polyhedron type
	Click on Create to build a Platonic Solid
*/

	
// This class represents the main class of the script
function Create3DPlatonicSolid()
{
	// Variable used to keep track of 'this' reference
	var create3DPlatonicSolid = this;
	
	// Create an instance of the utils class to use its functions
	var utils = new Create3DPlatonicSolidUtils();

	// Script infos
	this.scriptMinSupportVersion = "9.0";
	this.scriptName = "Create3DPlatonicSolid.jsx";	
	this.scriptVersion = "1.0";
	this.scriptTitle = "Create 3D Platonic Solid";
	this.scriptCopyright = "Copyright (c) 2010 Charles Bordenave";
	this.scriptHomepage = "http://www.nabscripts.com";
	this.scriptDescription = {en:"This script creates 3D regular convex polyhedra known as Platonic Solids.", 
							  fr:"Ce script crée des polyèdres réguliers convexes 3D connus sous le nom de Solides de Platon."};
	this.scriptAbout = {en:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription), 
						fr:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription)};		
	this.scriptUsage = {en:	"\u25BA In After Effects CS4 or later, run the script \\r" +
							"\u25BA Specify the type and size of the polyhedron \\r" +
							"\u25BA Click on Create to build a Platonic Solid",
						fr:	"\u25BA Dans After Effects CS4 ou supérieur, exécuter le script \\r" +
							"\u25BA Spécifier le type et la taille du polyèdre \\r" +
							"\u25BA Cliquer sur Créer pour construire un Solide de Platon"};

	// Errors
	this.requirementErr = {en:"This script requires After Effects CS4 or later.", fr:"Ce script nécessite After Effects CS4 ou supérieur."};	
	this.noCompErr = {en:"A comp must be active.", fr:"Une composition doit être active."};
	this.processErr = {en:"Error occurred during the process.", fr:"Une erreur est survenue lors de l'exécution."};
	
	// UI strings 
	this.aboutBtnName = "?";
	this.polyhedronStName = {en:"Polyhedron:", fr:"Polyhèdre:"};
	this.polyhedronLstChoices = {en:"['Tetrahedron', 'Hexahedron (cube)', 'Octahedron', 'Dodecahedron', 'Icosahedron']", fr:"['Tétrahèdre', 'Hexahèdre (cube)', 'Octahèdre', 'Dodécahèdre', 'Icosahèdre']"};
	this.scaleFactorStName = {en:"Size:", fr:"Taille:"};
	this.runBtnName = {en:"Create", fr:"Créer"};
	
	// Default values
	this.polyhedronLstSelDflt = 0;
	this.scaleFactorEtDflt = 500;
	
	// Internal data
	this.polyhedronLstVal = this.polyhedronLstSelDflt;
	this.scaleFactor = this.scaleFactorEtDflt;
	
	// Creates and displays the script interface
	this.buildUI = function (thisObj)
	{
		// dockable panel or palette
		var pal = (thisObj instanceof Panel) ? thisObj : new Window("palette", this.scriptTitle, undefined, {resizeable:false});
		
		// resource specifications
		var res =
		"group { orientation:'column', alignment:['fill','fill'], alignChildren:['right','top'], \
			gr1: Group { \
				aboutBtn: Button { text:'" + this.aboutBtnName + "', preferredSize:[25,20] } \
			}, \
			gr2: Group { orientation:'row', \
				polyhedronSt: StaticText { text:'" + utils.loc(this.polyhedronStName) + "' }, \
				polyhedronLst: DropDownList { properties:{items:" + utils.loc(this.polyhedronLstChoices) + "} } \
			}, \
			gr3: Group { orientation:'row', \
				scaleFactorSt: StaticText { text:'" + utils.loc(this.scaleFactorStName) + "' }, \
				scaleFactorEt: EditText { text:'" + this.scaleFactorEtDflt + "', characters:4 }, \
				fooSt: StaticText { } \
			}, \
			gr4: Group { orientation:'row', alignment:['fill','top'], \
				runBtn: Button { text:'" + utils.loc(this.runBtnName) + "', alignment:['fill','center'] } \
			} \
		}"; 
		pal.gr = pal.add(res);
		
		pal.layout.layout(true);
		
		pal.gr.gr2.polyhedronLst.selection = this.polyhedronLstSelDflt;
		pal.gr.gr3.fooSt.size.width = pal.gr.gr2.polyhedronLst.size.width - pal.gr.gr3.scaleFactorEt.size.width - pal.gr.gr3.spacing;
		
		// event callbacks				
		pal.gr.gr1.aboutBtn.onClick = function () 
		{ 
			utils.createAboutDlg(create3DPlatonicSolid.scriptAbout, create3DPlatonicSolid.scriptUsage); 
		};

		pal.gr.gr2.polyhedronLst.onChange = function ()
		{
			create3DPlatonicSolid.polyhedronLstVal = this.selection.index;						
		};
		
		pal.gr.gr3.scaleFactorEt.onChange = function ()
		{
			if (isNaN(this.text) || parseFloat(this.text) <= 0) this.text = this.scaleFactorEtDflt;
			create3DPlatonicSolid.scaleFactor = parseFloat(this.text);
		};
				
		pal.gr.gr4.runBtn .onClick = function () 
		{ 
			create3DPlatonicSolid.create();
		};		
		
		// show user interface
		if (pal instanceof Window)
		{
			pal.center();
			pal.show();
		}
		else
		{
			pal.layout.layout(true);
		}	   
	};

	// Return true if the active item is not a composition, False otherwise
	this.checkActiveItem = function () 
	{
		return !(app.project.activeItem instanceof CompItem);
	};   
	
	this.createFace = function (comp, nullL, face)
	{
		var p0 = face[0]; // point [x,y,z]
		var p1 = face[1];
		var p2 = face[2];		
		
		var posVal = [0,0,0];
		var shapeW, shapeH;
		
		var numVerts = face.length;
		for (var i = 0; i < numVerts; i++)
		{
			posVal += face[i];					
		}
		posVal /= numVerts;
		
		switch (numVerts) 
		{
			case 3:
				shapeW = utils.Length(face[1],face[2]);
				shapeH = (4/3)*utils.Length(face[0],(face[1]+face[2])/2);
				shapeW = Math.max(shapeW,shapeH);
				break;
			case 4:
				shapeW = utils.Length(face[0],face[1]);
				shapeH = shapeW;
				break;
			case 5:
			 	shapeW = utils.Length(face[2],face[3]);
			 	shapeH = (5/3)*utils.Length(face[0],(face[2]+face[3])/2);
			 	shapeW = Math.max(shapeW,shapeH);
				break;
			default:
				break;
		}
		posVal += [comp.width/2,comp.height/2,0];
		//posVal[1] *= -1; // y-axis points downwards in AE
		
		var n = utils.cross(p1-p0,p2-p0);
		var oriExpr = "lookAt(toWorld(anchorPoint), toWorld(anchorPoint) + [" + n[0] + "," + n[1] + "," + n[2] + "])";
		
		var w = Math.ceil(shapeW);
		var h = Math.ceil(shapeH);
		
		var layer = comp.layers.addSolid([1,1,1], "Face " + (Math.random()*100), w, h, comp.pixelAspect);
		layer.threeDLayer = true;
		
		layer.position.setValue(posVal);
		layer.orientation.expression = oriExpr;
		layer.orientation.setValue(layer.orientation.valueAtTime(comp.time,false));
		layer.orientation.expression = "";
		
		// add mask when odd number of vertices
		if (face.length % 2 == 1)
		{
			var shape = new Shape();
			var verts = [];
			
			for (i = 0; i < numVerts; i++)
			{
				nullL.position.expression =
				"L = thisComp.layer(\"" + layer.name + "\");\r" +
				"L.fromWorld([" + (face[i][0]+comp.width/2) + "," + (face[i][1]+comp.height/2) + "," + face[i][2] + "]);";
				
				var x = nullL.position.valueAtTime(comp.time,false)[0];
				var y = nullL.position.valueAtTime(comp.time,false)[1];
				
				verts.push([x,y]);//+[-comp.width/2,comp.height/2]);	
			}
			
			nullL.position.expression = "";
			
			shape.vertices = verts;
			shape.closed = true;
			
			var mask = layer.Masks.addProperty("Mask");
			mask.maskShape.setValue(shape);
		}
		layer.source.name = "Face";
		layer.name = "Face";
		layer.selected = false;
		
		return layer; 	
	};
	
	this.createPolyhedron = function (comp, facesList)
	{
		var nullL = comp.layers.addNull();
		nullL.name = "Controller";
		nullL.threeDLayer = true;
		
		for (var i = 0; i < facesList.length; i++)
		{
			var layer = this.createFace(comp, nullL, facesList[i]);
			layer.parent = nullL;
		}
		nullL.moveToBeginning();
		//nullL.remove(); 
	};
	
	// Functional part of the script
	this.create = function ()
	{
		try
		{
			var comp = app.project.activeItem;
			var err = this.noCompErr;
			if (this.checkActiveItem(comp)) throw(err);
		   				
			app.beginUndoGroup(this.scriptTitle);
			
			switch (this.polyhedronLstVal)
			{
				case 0: // 4 faces
					var a = 0.5 * this.scaleFactor;
					this.createPolyhedron(comp, 
									   [[[a,a,a],[-a,a,-a],[a,-a,-a]], 
									   [[-a,a,-a],[-a,-a,a],[a,-a,-a]], 
									   [[a,a,a],[a,-a,-a],[-a,-a,a]], 
									   [[a,a,a],[-a,-a,a],[-a,a,-a]]]);
					break;
				case 1: // 6 faces
					var a = 0.5 * this.scaleFactor;
					this.createPolyhedron(comp,
										[[[-a, -a, -a],   [a, -a, -a],	[a, -a,  a],   [-a, -a,  a]],
										 [[-a, -a, -a],   [-a, -a,  a],   [-a,  a,  a],   [-a,  a, -a]],
										 [[-a, -a,  a],   [ a, -a,  a],	[a,  a,  a],   [-a,  a,  a]],
										 [[-a,  a, -a],   [-a,  a,  a],	[a,  a,  a],	[a,  a, -a]],
										 [[a, -a, -a],	[a,  a, -a],	[a,  a,  a],	[a, -a,  a]],
										 [[-a, -a, -a],   [-a,  a, -a],	[a, a, -a],	[a, -a, -a]]]);
					break;
				case 2: // 8 faces
					var a = (1 / (2 * Math.sqrt(2))) * this.scaleFactor; 
					var b = 0.5*this.scaleFactor;
					this.createPolyhedron(comp,
										[[[-a,0,a],   [-a,0,-a],	[0,b,0]],
										[[-a,0,-a],   [a,0,-a],	[0,  b,  0]],
										[[a,  0, -a],   [a,  0,  a],	[0,  b,  0]],
										[[a,  0,  a],   [-a,  0,  a],	[0,  b,  0]],
										[[a,  0, -a],   [-a,  0, -a],	[0, -b,  0]],
										[[-a,  0, -a],   [-a,  0,  a],	[0, -b,  0]],
										[[a,  0,  a],   [a,  0, -a],	[0, -b,  0]],
										[[-a,  0,  a],	[a,  0,  a],	[0, -b,  0]]]);
					break;
				case 3: // 12 faces
					var phi = (1 + Math.sqrt(5)) / 2;
					var b = 0.5 * (1 / phi) * this.scaleFactor;
					var c = 0.5 * (2 - phi) * this.scaleFactor; 
					var h = 0.5 * this.scaleFactor;
					this.createPolyhedron(comp,
										[[[c,0,h],[-c,  0,  h],[-b , b,  b],[0,  h , c],[b,  b , b]],
										[[-c,0,  h],[c,  0 , h],[b ,-b,  b],[0, -h , c],[-b ,-b , b]],
										[[c,0, -h],[-c,  0, -h],[-b ,-b ,-b],[0 ,-h, -c],[b ,-b, -b]],
										[[-c,0, -h],[c,  0 ,-h],[b , b ,-b],[0,  h, -c],[-b , b, -b]],
										[[0,h, -c],[0,  h  ,c],[b , b , b],[h,  c,  0],[b , b, -b]],
										[[0,h,  c],[0,  h ,-c],[-b , b ,-b],[-h , c , 0],[-b , b , b]],
										[[0,-h, -c],[0, -h , c],[-b, -b , b],[-h ,-c,  0],[-b, -b, -b]],
										[[0,-h,  c],[0, -h ,-c],[b ,-b, -b],[h ,-c,  0],[b, -b,  b]],
										[[h, c,  0],[h, -c , 0],[b ,-b,  b],[c , 0 , h],[b , b , b]],
										[[h, -c,  0],[h,  c , 0],[b , b, -b],[c,  0 ,-h],[b, -b, -b]],
										[[-h,  c,  0],[-h, -c,  0],[-b, -b, -b],[-c , 0, -h],[-b,  b, -b]],
										[[-h, -c,  0],[-h,  c,  0],[-b,  b , b],[-c , 0,  h],[-b ,-b,  b]]]);
					break;					
				case 4: // 20 faces
					var phi = (1 + Math.sqrt(5)) / 2;
					var a = 0.5*this.scaleFactor;
					var b = (1 / (2 * phi))*this.scaleFactor;
					this.createPolyhedron(comp,
										[[[0,b,-a], [b,a,0],  [-b,a,0]],
										 [[0,b,a], [-b,a,0], [b,a,0]],
										 [[0,b,a], [0,-b,a], [-a,0,b]],
										 [[0,b,a], [a,0,b], [0,-b,a]],
										 [[0,b,-a], [0,-b,-a], [a,0,-b]],
										 [[0,b,-a], [-a,0,-b], [0,-b,-a]],
										 [[0,-b,a], [b,-a,0], [-b,-a,0]],
										 [[0,-b,-a], [-b,-a,0], [b,-a,0]],
										[[-b,a,0], [-a,0,b], [-a,0,-b]],
										[[-b,-a,0], [-a,0,-b], [-a,0,b]],
										 [[b,a,0], [a,0,-b], [a,0,b]],
										 [[b,-a,0], [a,0,b], [a,0,-b]],
										 [[0,b,a], [-a,0,b], [-b,a,0]],
										 [[0,b,a], [b,a,0], [a,0,b]],
										 [[0,b,-a], [-b,a,0], [-a,0,-b]],
										 [[0,b,-a], [a,0,-b], [b,a,0]],
										 [[0,-b,-a], [-a,0,-b], [-b,-a,0]],
										 [[0,-b,-a], [b,-a,0], [a,0,-b]],
										 [[0,-b,a], [-b,-a,0], [-a,0,b]],
										 [[0,-b,a], [a,0,b], [b,-a,0]]]);					
					break;
				default:
					break;
			}			 
			app.endUndoGroup();	
		}
		catch(err)
		{
			utils.throwErr(err);
		}				
	};
	
	// Runs the script  
	this.run = function (thisObj) 
	{
		if (parseFloat(app.version) < parseFloat(this.scriptMinSupportVersion))
		{
			utils.throwErr(this.requirementErr);
		}
		else
		{
			this.buildUI(thisObj);
		}	
	};
}


// This class provides some utility functions
function Create3DPlatonicSolidUtils()
{
	// Variable used to keep track of 'this' reference
	var utils = this;
	
	// String localization function: english and french languages are supported
	this.loc = function (str)
	{
		return app.language == Language.FRENCH ? str.fr : str.en;
	};

	// Displays a window containg a localized error message
	this.throwErr = function (err)
	{
		var wndTitle = $.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf("."));
		Window.alert("Script error:\r" + this.loc(err), wndTitle, true);
	};			
	
	this.Length = function (u,v)
	{
		var res = 0;
		
		if (u.length == v.length)
		{
			for (var i = 0; i < v.length; i++)
			{
				res += Math.pow(v[i]-u[i],2);
			}
			res = Math.sqrt(res);
		}		
		
		return res;
	};
	
	this.cross = function (u,v)
	{
		var w0 = u[1]*v[2] - u[2]*v[1];
		var w1 = u[2]*v[0] - u[0]*v[2];
		var w2 = u[0]*v[1] - u[1]*v[0];
		return [w0,w1,w2];
	};
	
	// Displays a dialog containg the script description and usage
	this.createAboutDlg = function (aboutStr, usageStr)
	{	
		eval(unescape('%09%09%76%61%72%20%64%6C%67%20%3D%20%6E%65%77%20%57%69%6E%64%6F%77%28%22%64%69%61%6C%6F%67%22%2C%20%22%41%62%6F%75%74%22%29%3B%0A%09%20%20%20%20%20%20%09%20%20%20%20%20%20%20%09%0A%09%20%20%20%20%76%61%72%20%72%65%73%20%3D%0A%09%09%22%67%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%63%6F%6C%75%6D%6E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%61%6C%69%67%6E%43%68%69%6C%64%72%65%6E%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%09%09%09%70%6E%6C%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%62%65%64%70%61%6E%65%6C%27%2C%20%5C%0A%09%09%09%09%61%62%6F%75%74%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%44%65%73%63%72%69%70%74%69%6F%6E%27%2C%20%5C%0A%09%09%09%09%09%61%62%6F%75%74%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%2C%20%5C%0A%09%09%09%09%75%73%61%67%65%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%55%73%61%67%65%27%2C%20%5C%0A%09%09%09%09%09%75%73%61%67%65%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%20%5C%0A%09%09%09%7D%2C%20%5C%0A%09%09%09%62%74%6E%73%3A%20%47%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%72%6F%77%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%62%6F%74%74%6F%6D%27%5D%2C%20%5C%0A%09%09%09%09%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%74%68%65%72%20%53%63%72%69%70%74%73%2E%2E%2E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%6C%65%66%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%2C%20%5C%0A%09%09%09%09%6F%6B%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%6B%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%09%09%09%7D%20%5C%0A%09%09%7D%22%3B%20%0A%09%09%64%6C%67%2E%67%72%20%3D%20%64%6C%67%2E%61%64%64%28%72%65%73%29%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%3B%0A%09%09%7D%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%27%2F%67%2C%20%22%27%22%29%3B%0A%09%09%7D%3B%0A%09%09%09%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%76%61%72%20%63%6D%64%20%3D%20%22%22%3B%0A%09%09%09%76%61%72%20%75%72%6C%20%3D%20%22%68%74%74%70%3A%2F%2F%61%65%73%63%72%69%70%74%73%2E%63%6F%6D%2F%63%61%74%65%67%6F%72%79%2F%73%63%72%69%70%74%73%2F%6E%61%62%2F%22%3B%0A%09%0A%09%09%09%69%66%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%0A%09%09%09%7B%0A%09%20%20%20%20%20%20%20%20%09%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%20%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%49%6E%74%65%72%6E%65%74%20%45%78%70%6C%6F%72%65%72%2F%69%65%78%70%6C%6F%72%65%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%7D%0A%09%09%09%65%6C%73%65%0A%09%09%09%09%63%6D%64%20%2B%3D%20%22%6F%70%65%6E%20%5C%22%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%20%20%20%20%20%20%20%20%20%0A%09%09%09%74%72%79%0A%09%09%09%7B%0A%09%09%09%09%73%79%73%74%65%6D%2E%63%61%6C%6C%53%79%73%74%65%6D%28%63%6D%64%29%3B%0A%09%09%09%7D%0A%09%09%09%63%61%74%63%68%28%65%29%0A%09%09%09%7B%0A%09%09%09%09%61%6C%65%72%74%28%65%29%3B%0A%09%09%09%7D%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%6B%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%0A%09%09%7B%0A%09%09%09%64%6C%67%2E%63%6C%6F%73%65%28%29%3B%20%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%63%65%6E%74%65%72%28%29%3B%0A%09%09%64%6C%67%2E%73%68%6F%77%28%29%3B'));		
	};	
}


// Creates an instance of the main class and run it
new Create3DPlatonicSolid().run(this);
